<?php
error_reporting(E_ALL);
ini_set('display_errors','0');
ini_set('log_errors','1');
ini_set('error_log', __DIR__ . '/php_error.log');

header('Content-Type: application/json');
require 'config.php';

try {
    session_start();
    if (empty($_SESSION['id'])) {
        throw new Exception('Not authenticated.');
    }

    $book_id = intval($_POST['book_id'] ?? 0);
    $add_qty = intval($_POST['add_qty'] ?? 0);

    if ($book_id < 1 || $add_qty < 1) {
        throw new Exception('Invalid parameters.');
    }

    $qrDir = __DIR__ . '/uploads/book_qr/';
    if (!is_dir($qrDir) && !mkdir($qrDir, 0777, true)) {
        throw new Exception("Cannot create directory: $qrDir");
    }

    $pdo->beginTransaction();

    $stmt = $pdo->prepare("SELECT stock FROM books WHERE id = ? FOR UPDATE");
    $stmt->execute([$book_id]);
    $current_stock = $stmt->fetchColumn();
    if ($current_stock === false) {
        $pdo->rollBack();
        throw new Exception("Book not found.");
    }
    $current_stock = intval($current_stock);

    $stmt = $pdo->prepare("SELECT COALESCE(MAX(CAST(SUBSTRING_INDEX(copy_code, '-', -1) AS UNSIGNED)), 0) AS lastnum FROM book_copies WHERE book_id = ?");
    $stmt->execute([$book_id]);
    $lastnum = intval($stmt->fetchColumn() ?? 0);

    $new_stock_total = $current_stock + $add_qty;

    $upd = $pdo->prepare("UPDATE books SET stock = :stock WHERE id = :id");
    $upd->execute([
        ':stock' => $new_stock_total,
        ':id'    => $book_id
    ]);

    $ins = $pdo->prepare("INSERT INTO book_copies (book_id, copy_code, code_type, code_image, status) VALUES (:book, :code, 'QR', :img, 'available')");

    for ($i = 1; $i <= $add_qty; $i++) {
        $seq = $lastnum + $i;
        $copyCode = $book_id . '-' . str_pad($seq, 4, '0', STR_PAD_LEFT);
        $qrFile = "qr_{$copyCode}.png";
        $qrPath = $qrDir . $qrFile;
        $apiUrl = 'https://api.qrserver.com/v1/create-qr-code/' . '?data=' . urlencode($copyCode) . '&size=200x200';

        $ch = curl_init($apiUrl);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT        => 20,
        ]);
        $qrData = curl_exec($ch);
        $err    = curl_error($ch);
        curl_close($ch);

        if ($err || !$qrData) {
            $pdo->rollBack();
            throw new Exception("Failed to generate QR for {$copyCode}: {$err}");
        }

        if (file_put_contents($qrPath, $qrData) === false) {
            $pdo->rollBack();
            throw new Exception("Failed to save QR file: {$qrPath}");
        }

        $ins->execute([
            ':book' => $book_id,
            ':code' => $copyCode,
            ':img'  => 'uploads/book_qr/' . $qrFile
        ]);
    }

    $pdo->commit();

    echo json_encode([
        'status' => 'success',
        'message' => "Stock updated: +{$add_qty} copies added. New total: {$new_stock_total}."
    ]);
    exit;

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("update_stock.php Error: " . $e->getMessage());
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
    exit;
} ?>