<?php
header('Content-Type: application/json; charset=utf-8');
require 'config.php';
session_start();

try {
    if (empty($_SESSION['id'])) throw new Exception('Unauthorized');

    $student_id = isset($_POST['student_id']) ? (int)$_POST['student_id'] : 0;
    $assigned_ids = $_POST['assigned_ids'] ?? [];
    $action = isset($_POST['action']) ? trim($_POST['action']) : 'submit'; 
    $actual_date_raw = trim($_POST['actual_date'] ?? '');
    if (!$student_id) throw new Exception('Invalid student id');
    if (empty($assigned_ids) || !is_array($assigned_ids)) throw new Exception('No assigned books selected');
    if (!$actual_date_raw) throw new Exception('Select date & time');

    $actual_date = date('Y-m-d H:i:s', strtotime($actual_date_raw));
    if (!$actual_date) throw new Exception('Invalid date/time');

    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $statusToSet = $action === 'lost' ? 'lost' : 'submitted';
    $copyStatus = $action === 'lost' ? 'lost' : 'available';

    $processed = [];
    $failed = [];
    $lostBookNames = []; 

    $pdo->beginTransaction();

    foreach ($assigned_ids as $aidRaw) {
        $aid = (int)$aidRaw;
        if (!$aid) { $failed[] = "Invalid assigned id: {$aidRaw}"; continue; }

        $s = $pdo->prepare("SELECT id, book_id, pickup_date, supposed_submit_date FROM student_assigned_books WHERE id = :id AND student_id = :sid LIMIT 1");
        $s->execute([':id' => $aid, ':sid' => $student_id]);
        $row = $s->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            $failed[] = "Assigned id {$aid} not found for this student";
            continue;
        }

        $copy_code = $row['book_id'];

        $u1 = $pdo->prepare("UPDATE student_assigned_books SET actual_submit_date = :adate, status = :status WHERE id = :id");
        $u1->execute([':adate' => $actual_date, ':status' => $statusToSet, ':id' => $aid]);

        $u2 = $pdo->prepare("UPDATE book_copies SET status = :cstatus WHERE copy_code = :cc");
        $u2->execute([':cstatus' => $copyStatus, ':cc' => $copy_code]);

        $q = $pdo->prepare("SELECT b.name FROM book_copies bc JOIN books b ON bc.book_id = b.id WHERE bc.copy_code = :cc LIMIT 1");
        $q->execute([':cc' => $copy_code]);
        $bookRow = $q->fetch(PDO::FETCH_ASSOC);
        if ($bookRow && !empty($bookRow['name'])) {
            if ($action === 'lost') $lostBookNames[] = $bookRow['name'];
        }

        $processed[] = $aid;
    }

    $pdo->commit();

    $mailResult = null;
    if ($action === 'lost' && !empty($lostBookNames)) {
        $q = $pdo->prepare("SELECT name, email FROM students_temp WHERE id = :id LIMIT 1");
        $q->execute([':id' => $student_id]);
        $stu = $q->fetch(PDO::FETCH_ASSOC);

        $to = $stu['email'] ?? '';
        $studentName = $stu['name'] ?? '';
        $adminSchool = isset($_SESSION['school']) ? $_SESSION['school'] : '';

        if ($to && filter_var($to, filter: FILTER_VALIDATE_EMAIL)) {
            $subject = "Notice: Lost Book(s) - {$adminSchool}";
            $bookListHtml = '<ul>';
            foreach ($lostBookNames as $bn) {
                $bookListHtml .= '<li>' . htmlspecialchars($bn, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8') . '</li>';
            }
            $bookListHtml .= '</ul>';

            $message = "
                <html>
                <head><meta charset='utf-8'><title>Lost Book Notification</title></head>
                <body style='font-family:Arial,Helvetica,sans-serif;color:#222;line-height:1.4'>
                  <h3 style='color:#c0392b;'>Important Notice from {$adminSchool}</h3>
                  <p>Dear Parent / Guardian,</p>
                  <p>We regret to inform you that the following book(s) assigned to <strong>" . htmlspecialchars($studentName, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8') . "</strong> have been recorded as <strong>lost</strong> on <strong>" . date('Y-m-d H:i:s') . "</strong>:</p>
                  {$bookListHtml}
                  <p>Please contact the school library as soon as possible to discuss the replacement or compensation procedure.</p>
                  <p>Regards,<br><strong>{$adminSchool} Library</strong></p>
                </body>
                </html>";

            $fromEmail = 'no-reply@' . ($_SERVER['SERVER_NAME'] ?? 'localhost');
            $headers = "MIME-Version: 1.0\r\n";
            $headers .= "Content-type: text/html; charset=UTF-8\r\n";
            $headers .= "From: {$adminSchool} <{$fromEmail}>\r\n";
            $headers .= "Reply-To: {$fromEmail}\r\n";

            $mailResult = mail($to, $subject, $message, $headers);
        } else {
            $mailResult = false;
        }
    }

    $msgParts = [];
    if (!empty($processed)) $msgParts[] = count($processed) . ' record(s) updated';
    if (!empty($failed)) $msgParts[] = 'Failed: ' . implode('; ', $failed);
    if ($action === 'lost') {
        $msgParts[] = 'Email ' . ($mailResult ? 'sent' : 'not sent');
    }

    echo json_encode(['status' => 'success', 'message' => implode(' | ', $msgParts), 'processed' => $processed, 'failed' => $failed, 'mail_sent' => $mailResult]);
    exit;

} catch (Exception $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    exit;
} ?>