<?php
// Disable error display to prevent HTML in JSON response
ini_set('display_errors', 0);
error_reporting(0);

// Enable error logging for debugging
ini_set('log_errors', 1);
ini_set('error_log', '/path/to/error.log'); // Set appropriate path

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Access-Control-Allow-Headers, Content-Type, Access-Control-Allow-Methods, Authorization, X-Requested-With');

// Error handler to ensure JSON response
function handleError($errno, $errstr, $errfile, $errline) {
    error_log("PHP Error: [$errno] $errstr in $errfile on line $errline");
    respond(false, 'Internal server error');
}
set_error_handler('handleError');

// Exception handler
function handleException($exception) {
    error_log("PHP Exception: " . $exception->getMessage());
    respond(false, 'Internal server error');
}
set_exception_handler('handleException');

function respond($status, $message = '', $extra = []) {
    // Clean any output buffer to prevent HTML before JSON
    if (ob_get_level()) {
        ob_clean();
    }
    
    header('Content-Type: application/json');
    echo json_encode(array_merge(['status' => $status, 'message' => $message], $extra));
    exit;
}

try {
    // Check if config file exists
    if (!file_exists('../configDealer.php')) {
        respond(false, 'Configuration file not found');
    }
    
    include '../configDealer.php'; // gives $conn (mysqli)
    
    // Check database connection
    if (!isset($conn) || $conn->connect_error) {
        error_log("Database connection failed: " . ($conn->connect_error ?? 'Connection variable not set'));
        respond(false, 'Database connection failed');
    }
    
    date_default_timezone_set('Asia/Kolkata');
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        respond(false, 'Method Not Allowed');
    }
    
    $student_id = trim($_POST['student_id'] ?? '');
    if (empty($student_id)) {
        respond(false, 'student_id is required');
    }
    
    // Validate student_id is numeric
    if (!is_numeric($student_id)) {
        respond(false, 'Invalid student_id format');
    }
    
    // Check if student exists
    $check_stmt = $conn->prepare("SELECT id FROM students_temp WHERE id = ?");
    if (!$check_stmt) {
        error_log("Prepare failed: " . $conn->error);
        respond(false, 'Database error');
    }
    
    $check_stmt->bind_param('i', $student_id);
    $check_stmt->execute();
    $result = $check_stmt->get_result();
    
    if ($result->num_rows === 0) {
        $check_stmt->close();
        respond(false, 'Student not found');
    }
    $check_stmt->close();
    
    // Collect fields
    $fields = [
        'name','email','phone','address','city','state','country','pin',
        'dob','age','applied_class','prev_school','prev_class',
        'marks_got','out_off','percentage','learning_language',
        'preferred_language','applied_at','approved_at','password'
    ];
    
    $updates = [];
    $params  = [];
    $types   = '';
    
    foreach ($fields as $f) {
        if (isset($_POST[$f]) && $_POST[$f] !== '') {
            if ($f === 'password' && !empty($_POST[$f])) {
                $hashed = password_hash($_POST[$f], PASSWORD_BCRYPT);
                $updates[] = "`$f` = ?";
                $params[]  = $hashed;
                $types    .= 's';
            } else {
                $updates[] = "`$f` = ?";
                $params[]  = trim($_POST[$f]);
                $types    .= 's';
            }
        }
    }
    
    // Handle image upload
    $imagePath = null;
    if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = '../../Student/uploads/students';
        
        // Create directory if it doesn't exist
        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0777, true)) {
                error_log("Failed to create upload directory: $uploadDir");
                respond(false, 'Failed to create upload directory');
            }
        }
        
        // Check if directory is writable
        if (!is_writable($uploadDir)) {
            error_log("Upload directory is not writable: $uploadDir");
            respond(false, 'Upload directory is not writable');
        }
        
        $tmp     = $_FILES['image']['tmp_name'];
        $nameImg = basename($_FILES['image']['name']);
        $safe    = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $nameImg);
        $target  = "$uploadDir/" . time() . "_$safe";
        
        // Validate file type
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $fileType = $_FILES['image']['type'];
        
        if (!in_array($fileType, $allowedTypes)) {
            respond(false, 'Invalid file type. Only JPEG, PNG, and GIF are allowed.');
        }
        
        // Check file size (5MB limit)
        if ($_FILES['image']['size'] > 5 * 1024 * 1024) {
            respond(false, 'File size too large. Maximum 5MB allowed.');
        }
        
        if (move_uploaded_file($tmp, $target)) {
            $imagePath = 'uploads/students/' . basename($target);
            $updates[] = "`image` = ?";
            $params[]  = $imagePath;
            $types    .= 's';
        } else {
            error_log("Failed to move uploaded file from $tmp to $target");
            respond(false, 'Failed to upload image');
        }
    } elseif (isset($_FILES['image']) && $_FILES['image']['error'] !== UPLOAD_ERR_NO_FILE) {
        // Handle upload errors
        $uploadErrors = [
            UPLOAD_ERR_INI_SIZE => 'File too large (PHP ini limit)',
            UPLOAD_ERR_FORM_SIZE => 'File too large (form limit)',
            UPLOAD_ERR_PARTIAL => 'File partially uploaded',
            UPLOAD_ERR_NO_TMP_DIR => 'No temporary directory',
            UPLOAD_ERR_CANT_WRITE => 'Cannot write to disk',
            UPLOAD_ERR_EXTENSION => 'Upload stopped by extension'
        ];
        
        $errorMsg = $uploadErrors[$_FILES['image']['error']] ?? 'Unknown upload error';
        error_log("File upload error: " . $errorMsg);
        respond(false, $errorMsg);
    }
    
    if (empty($updates)) {
        respond(false, 'No fields to update');
    }
    
    $updates_sql = implode(', ', $updates);
    $params[]    = $student_id;
    $types      .= 'i';
    
    $sql  = "UPDATE students_temp SET $updates_sql WHERE id = ?";
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        respond(false, 'Database prepare error');
    }
    
    if (!$stmt->bind_param($types, ...$params)) {
        error_log("Bind param failed: " . $stmt->error);
        $stmt->close();
        respond(false, 'Database bind error');
    }
    
    if ($stmt->execute()) {
        if ($stmt->affected_rows > 0) {
            respond(true, 'Profile updated successfully');
        } else {
            respond(false, 'No changes made to profile');
        }
    } else {
        error_log("Execute failed: " . $stmt->error);
        respond(false, 'Update failed: Database error');
    }
    
    $stmt->close();
    $conn->close();

} catch (Exception $e) {
    error_log("Exception in edit_profile.php: " . $e->getMessage());
    respond(false, 'Internal server error');
}
?>