<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    exit(0);
}

class GeofencingAPI {
    
    // Define your designated areas here
    private $designatedAreas = [
        'campus' => [
            'name' => 'University Campus',
            'type' => 'polygon',
            'coordinates' => [
                ['lat' => 22.5726, 'lng' => 88.3639], // Point 1
                ['lat' => 22.5730, 'lng' => 88.3645], // Point 2
                ['lat' => 22.5735, 'lng' => 88.3642], // Point 3
                ['lat' => 22.5732, 'lng' => 88.3635], // Point 4
                ['lat' => 22.5728, 'lng' => 88.3638], // Point 5
            ]
        ],
        'library' => [
            'name' => 'Library Zone',
            'type' => 'circle',
            'center' => ['lat' => 22.5728, 'lng' => 88.3640],
            'radius' => 100 // radius in meters
        ],
        'office' => [
            'name' => 'Office Building',
            'type' => 'rectangle',
            'bounds' => [
                'north' => 22.5735,
                'south' => 22.5725,
                'east' => 88.3650,
                'west' => 88.3630
            ]
        ]
    ];

    public function checkGeofence() {
        try {
            // Get input parameters
            $input = $this->getInput();
            
            if (!$this->validateInput($input)) {
                return $this->errorResponse('Invalid input parameters', 400);
            }

            $lat = floatval($input['latitude']);
            $lng = floatval($input['longitude']);
            $areaId = $input['area_id'] ?? null;

            $results = [];

            if ($areaId && isset($this->designatedAreas[$areaId])) {
                // Check specific area
                $area = $this->designatedAreas[$areaId];
                $isInside = $this->isPointInArea($lat, $lng, $area);
                $results[] = [
                    'area_id' => $areaId,
                    'area_name' => $area['name'],
                    'is_inside' => $isInside,
                    'area_type' => $area['type']
                ];
            } else {
                // Check all areas
                foreach ($this->designatedAreas as $id => $area) {
                    $isInside = $this->isPointInArea($lat, $lng, $area);
                    $results[] = [
                        'area_id' => $id,
                        'area_name' => $area['name'],
                        'is_inside' => $isInside,
                        'area_type' => $area['type']
                    ];
                }
            }

            return $this->successResponse([
                'location' => [
                    'latitude' => $lat,
                    'longitude' => $lng
                ],
                'areas_checked' => count($results),
                'results' => $results,
                'timestamp' => date('Y-m-d H:i:s')
            ]);

        } catch (Exception $e) {
            return $this->errorResponse('Server error: ' . $e->getMessage(), 500);
        }
    }

    private function getInput() {
        $method = $_SERVER['REQUEST_METHOD'];
        
        if ($method === 'POST') {
            $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
            
            if (strpos($contentType, 'application/json') !== false) {
                $input = json_decode(file_get_contents('php://input'), true);
            } else {
                $input = $_POST;
            }
        } else if ($method === 'GET') {
            $input = $_GET;
        } else {
            throw new Exception('Method not allowed');
        }

        return $input;
    }

    private function validateInput($input) {
        if (!isset($input['latitude']) || !isset($input['longitude'])) {
            return false;
        }

        $lat = floatval($input['latitude']);
        $lng = floatval($input['longitude']);

        // Validate latitude (-90 to 90)
        if ($lat < -90 || $lat > 90) {
            return false;
        }

        // Validate longitude (-180 to 180)
        if ($lng < -180 || $lng > 180) {
            return false;
        }

        return true;
    }

    private function isPointInArea($lat, $lng, $area) {
        switch ($area['type']) {
            case 'polygon':
                return $this->isPointInPolygon($lat, $lng, $area['coordinates']);
            
            case 'circle':
                return $this->isPointInCircle($lat, $lng, $area['center'], $area['radius']);
            
            case 'rectangle':
                return $this->isPointInRectangle($lat, $lng, $area['bounds']);
            
            default:
                return false;
        }
    }

    private function isPointInPolygon($lat, $lng, $polygon) {
        $vertices = count($polygon);
        $inside = false;

        for ($i = 0, $j = $vertices - 1; $i < $vertices; $j = $i++) {
            $xi = $polygon[$i]['lng'];
            $yi = $polygon[$i]['lat'];
            $xj = $polygon[$j]['lng'];
            $yj = $polygon[$j]['lat'];

            if ((($yi > $lat) != ($yj > $lat)) && 
                ($lng < ($xj - $xi) * ($lat - $yi) / ($yj - $yi) + $xi)) {
                $inside = !$inside;
            }
        }

        return $inside;
    }

    private function isPointInCircle($lat, $lng, $center, $radius) {
        $distance = $this->getDistance($lat, $lng, $center['lat'], $center['lng']);
        return $distance <= $radius;
    }

    private function isPointInRectangle($lat, $lng, $bounds) {
        return ($lat >= $bounds['south'] && $lat <= $bounds['north'] &&
                $lng >= $bounds['west'] && $lng <= $bounds['east']);
    }

    private function getDistance($lat1, $lng1, $lat2, $lng2) {
        $earthRadius = 6371000; // Earth radius in meters

        $dLat = deg2rad($lat2 - $lat1);
        $dLng = deg2rad($lng2 - $lng1);

        $a = sin($dLat/2) * sin($dLat/2) +
             cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
             sin($dLng/2) * sin($dLng/2);

        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        $distance = $earthRadius * $c;

        return $distance;
    }

    private function successResponse($data) {
        return json_encode([
            'status' => true,
            'message' => 'Location checked successfully',
            'data' => $data
        ], JSON_PRETTY_PRINT);
    }

    private function errorResponse($message, $code = 400) {
        http_response_code($code);
        return json_encode([
            'status' => false,
            'message' => $message,
            'error_code' => $code
        ], JSON_PRETTY_PRINT);
    }

    public function getAreas() {
        $areas = [];
        foreach ($this->designatedAreas as $id => $area) {
            $areas[] = [
                'area_id' => $id,
                'name' => $area['name'],
                'type' => $area['type']
            ];
        }

        return $this->successResponse([
            'total_areas' => count($areas),
            'areas' => $areas
        ]);
    }
}

// Initialize API
$api = new GeofencingAPI();

// Route handling
$action = $_GET['action'] ?? 'check';

switch ($action) {
    case 'check':
        echo $api->checkGeofence();
        break;
    
    case 'areas':
        echo $api->getAreas();
        break;
    
    default:
        http_response_code(404);
        echo json_encode([
            'status' => false,
            'message' => 'Action not found',
            'available_actions' => ['check', 'areas']
        ], JSON_PRETTY_PRINT);
        break;
}
?>

<!-- 
API USAGE EXAMPLES:

1. Check if point is in specific area (POST):
POST /geofencing.php?action=check
Content-Type: application/json
{
    "latitude": 22.5730,
    "longitude": 88.3640,
    "area_id": "campus"
}

2. Check if point is in any area (GET):
GET /geofencing.php?action=check&latitude=22.5730&longitude=88.3640

3. Get all available areas:
GET /geofencing.php?action=areas

RESPONSE EXAMPLES:

Success Response:
{
    "status": true,
    "message": "Location checked successfully",
    "data": {
        "location": {
            "latitude": 22.573,
            "longitude": 88.364
        },
        "areas_checked": 3,
        "results": [
            {
                "area_id": "campus",
                "area_name": "University Campus",
                "is_inside": true,
                "area_type": "polygon"
            },
            {
                "area_id": "library",
                "area_name": "Library Zone",
                "is_inside": false,
                "area_type": "circle"
            }
        ],
        "timestamp": "2025-07-30 15:30:45"
    }
}

Error Response:
{
    "status": false,
    "message": "Invalid input parameters",
    "error_code": 400
}
-->