<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Access-Control-Allow-Headers, Content-Type, Access-Control-Allow-Methods, Authorization, X-Requested-With');

try {
    // Create PDO connection
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (!isset($input['user_id']) || empty($input['user_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'User ID is required']);
        exit();
    }
    
    $user_id = intval($input['user_id']);
    
    // Validate user exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    if (!$stmt->fetch()) {
        http_response_code(404);
        echo json_encode(['error' => 'User not found']);
        exit();
    }
    
    // Prepare update data
    $updateFields = [];
    $updateValues = [];
    
    // Map of allowed fields to update
    $allowedFields = [
        'name' => 'name',
        'phone' => 'phone',
        'email' => 'email',
        'address' => 'address',
        'university' => 'university',
        'college' => 'college',
        'degree' => 'degree',
        'duration' => 'duration',
        'school' => 'school',
        'position' => 'position',
        'city' => 'city'
    ];
    
    // Build update query for basic fields
    foreach ($allowedFields as $inputKey => $dbColumn) {
        if (isset($input[$inputKey])) {
            $updateFields[] = "$dbColumn = ?";
            $updateValues[] = $input[$inputKey];
        }
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // Update basic profile fields
        if (!empty($updateFields)) {
            $sql = "UPDATE teachers SET " . implode(', ', $updateFields) . " WHERE id = ?";
            $updateValues[] = $user_id;
            $stmt = $pdo->prepare($sql);
            $stmt->execute($updateValues);
        }
        
        // Handle subjects separately
        if (isset($input['subjects']) && is_array($input['subjects'])) {
            // Delete existing subjects
            $stmt = $pdo->prepare("DELETE FROM user_subjects WHERE user_id = ?");
            $stmt->execute([$user_id]);
            
            // Insert new subjects
            if (!empty($input['subjects'])) {
                $stmt = $pdo->prepare("INSERT INTO user_subjects (user_id, subject) VALUES (?, ?)");
                foreach ($input['subjects'] as $subject) {
                    if (!empty(trim($subject))) {
                        $stmt->execute([$user_id, trim($subject)]);
                    }
                }
            }
        }
        
        // Commit transaction
        $pdo->commit();
        
        // Fetch updated profile data
        $stmt = $pdo->prepare("
            SELECT 
                u.*,
                GROUP_CONCAT(us.subject) as subjects_string
            FROM users u
            LEFT JOIN user_subjects us ON u.id = us.user_id
            WHERE u.id = ?
            GROUP BY u.id
        ");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user) {
            // Convert subjects string to array
            $user['subjects'] = !empty($user['subjects_string']) 
                ? explode(',', $user['subjects_string']) 
                : [];
            unset($user['subjects_string']);
            
            // Remove sensitive data
            unset($user['password']);
            unset($user['firebase_token']);
            
            // Return success response
            echo json_encode([
                'success' => true,
                'message' => 'Profile updated successfully',
                'data' => $user
            ]);
        } else {
            throw new Exception('Failed to fetch updated profile');
        }
        
    } catch (Exception $e) {
        // Rollback transaction on error
        $pdo->rollback();
        throw $e;
    }
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Database error: ' . $e->getMessage()
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Server error: ' . $e->getMessage()
    ]);
}
?>